/*
 * Decompiled with CFR 0.152.
 */
package owl.automaton.algorithm.simulations;

import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import owl.automaton.Automaton;
import owl.automaton.Views;
import owl.automaton.acceptance.BuchiAcceptance;
import owl.automaton.algorithm.simulations.BackwardDirectSimulation;
import owl.automaton.algorithm.simulations.ColorRefinement;
import owl.automaton.algorithm.simulations.ForwardDelayedSimulation;
import owl.automaton.algorithm.simulations.ForwardDirectLookaheadSimulation;
import owl.automaton.algorithm.simulations.ForwardDirectSimulation;
import owl.automaton.algorithm.simulations.ForwardFairSimulation;
import owl.automaton.algorithm.simulations.SimulationGame;
import owl.automaton.algorithm.simulations.SimulationStates;
import owl.collections.Pair;
import owl.command.AutomatonConversionCommands;
import owl.game.algorithms.OinkGameSolver;
import owl.game.algorithms.ParityGameSolver;
import owl.game.algorithms.ZielonkaGameSolver;
import owl.translations.nbadet.NbaDet;

public final class BuchiSimulation {
    public static final Logger logger = Logger.getLogger(BuchiSimulation.class.getName());
    private final ParityGameSolver solver;

    public BuchiSimulation() {
        this.solver = OinkGameSolver.checkOinkExecutable() ? new OinkGameSolver() : new ZielonkaGameSolver();
    }

    public BuchiSimulation(ParityGameSolver pgSolver) {
        this.solver = pgSolver;
    }

    private static <S> Set<Pair<S, S>> makeTransitive(Set<Pair<S, S>> relation) {
        HashSet<Pair<S, S>> out = new HashSet<Pair<S, S>>(relation);
        boolean cont = true;
        while (cont) {
            HashSet toAdd = new HashSet();
            for (Pair pair : out) {
                for (Pair<S, S> r : relation) {
                    if (!pair.snd().equals(r.fst())) continue;
                    toAdd.add(Pair.of(pair.fst(), r.snd()));
                }
            }
            cont = out.addAll(toAdd);
        }
        return out;
    }

    public static <S> Set<Pair<S, S>> computeEquivalence(Set<Pair<S, S>> relation) {
        HashSet out = new HashSet();
        Set<Pair<S, S>> tc = BuchiSimulation.makeTransitive(relation);
        Set reverseTc = tc.stream().map(Pair::swap).collect(Collectors.toSet());
        tc.forEach(l -> reverseTc.forEach(r -> {
            if (l.equals(r)) {
                out.add((Pair)l);
            }
        }));
        for (Pair pair : out) {
            assert (out.contains(pair.swap()));
        }
        return out;
    }

    public static <S> Automaton<Set<S>, BuchiAcceptance> compute(Automaton<S, BuchiAcceptance> automaton, AutomatonConversionCommands.NbaSimCommand args) {
        Set<Pair<S, S>> rel;
        Map<Handler, Level> oldLogLevels = null;
        if (args.verboseFine()) {
            oldLogLevels = NbaDet.overrideLogLevel(Level.FINE);
            logger.setLevel(Level.FINE);
        }
        logger.fine("Starting simulation computation");
        OinkGameSolver solver = new OinkGameSolver();
        BuchiSimulation simulator = new BuchiSimulation(solver);
        int pebbles = args.pebbleCount();
        if (args.sanity()) {
            logger.fine("Starting sanity check for di < de < f on automaton with " + pebbles + " pebbles.");
            Set<Pair<S, S>> relDirect = simulator.directSimulation(automaton, automaton, pebbles);
            Set<Pair<S, S>> relDirectRefinement = ColorRefinement.of(automaton);
            Set<Pair<S, S>> relDelayed = simulator.delayedSimulation(automaton, automaton, pebbles);
            Set<Pair<S, S>> relFair = simulator.fairSimulation(automaton, automaton, pebbles);
            logger.fine("Direct simulation pairs: " + relDirect);
            logger.fine("Delayed simulation pairs: " + (Set)relDelayed);
            logger.fine("Fair simulation paris: " + relFair);
            if (!relDelayed.containsAll(relDirect)) {
                logger.severe("Delayed not superset of direct, misses following pairs:");
                logger.severe("\t" + Sets.difference(relDirect, relFair));
                throw new AssertionError((Object)"Delayed not superset of direct.");
            }
            if (!relFair.containsAll(relDelayed)) {
                logger.severe("Fair not superset of delayed, misses following pairs:");
                logger.severe("\t" + Sets.difference((Set)relDelayed, relFair));
                throw new AssertionError((Object)"Fair not superset of delayed.");
            }
            logger.fine("Checking color refinement implementation");
            if (!relDirectRefinement.containsAll(relDirect)) {
                logger.severe("directRefinement not superset of direct, diref does not contain:");
                logger.severe("\t" + Sets.difference(relDirect, relDirectRefinement));
                throw new AssertionError((Object)"diref not superset of di.");
            }
            if (!relDirect.containsAll(relDirectRefinement)) {
                logger.severe("direct not superset of directRefinement, di does not contain:");
                logger.severe("\t" + Sets.difference(relDirectRefinement, relDirect));
                throw new AssertionError((Object)"di not superset of diref.");
            }
            logger.fine("All sanity checks passed; #DI: " + relDirect.size() + ", #DE: " + relDelayed.size() + ", #F : " + relFair.size());
        }
        switch (args.simulationType()) {
            case DIRECT_SIMULATION: {
                logger.fine("Starting direct simulation with " + pebbles + " pebbles.");
                rel = simulator.directSimulation(automaton, automaton, pebbles);
                break;
            }
            case DIRECT_SIMULATION_COLOUR_REFINEMENT: {
                logger.fine("Computing direct simulation based on color refinement.");
                rel = ColorRefinement.of(automaton);
                break;
            }
            case DELAYED_SIMULATION: {
                logger.fine("Starting delayed simulation with " + pebbles + " pebbles.");
                rel = simulator.delayedSimulation(automaton, automaton, pebbles);
                break;
            }
            case FAIR_SIMULATION: {
                throw new IllegalArgumentException("Cannot use fair simulation as it we cannot construct a quotient automaton.");
            }
            case BACKWARD_SIMULATION: {
                logger.fine("Starting backward simulation with " + pebbles + " pebbles.");
                rel = simulator.backwardSimulation(automaton, automaton, pebbles);
                break;
            }
            case LOOKAHEAD_DIRECT_SIMULATION: {
                logger.fine("Starting direct simulation with lookahead " + args.maxLookahead());
                rel = simulator.directLookaheadSimulation(automaton, automaton, args.maxLookahead());
                break;
            }
            default: {
                throw new AssertionError((Object)"Unreachable.");
            }
        }
        Set<Pair<S, S>> equivRel = BuchiSimulation.computeEquivalence(rel);
        HashMap classMap = new HashMap();
        for (Object state : automaton.states()) {
            classMap.put(state, new HashSet());
            for (Pair<S, S> p : equivRel) {
                if (!state.equals(p.fst())) continue;
                ((Set)classMap.get(state)).add(p.snd());
            }
        }
        Automaton<Set<S>, BuchiAcceptance> quotient = Views.quotientAutomaton(automaton, classMap::get);
        logger.fine("Input had " + automaton.states().size() + " states, while output has " + quotient.states().size() + " states");
        if (null != oldLogLevels) {
            NbaDet.restoreLogLevel(oldLogLevels);
        }
        return quotient;
    }

    private <S> Set<Pair<S, S>> lookaheadSimulate(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int maxLookahead, LookaheadGameConstructor<S> gc) {
        assert (maxLookahead > 0);
        if (BuchiSimulation.automatonTrivial(left) || BuchiSimulation.automatonTrivial(right)) {
            return Set.of();
        }
        ConcurrentHashMap.KeySetView known = ConcurrentHashMap.newKeySet();
        ConcurrentHashMap.KeySetView seen = ConcurrentHashMap.newKeySet();
        Set<SimulationStats> stats = Pair.allPairs(left.states(), right.states()).stream().map(pair -> {
            if (seen.add(pair)) {
                long startTime = System.currentTimeMillis();
                SimulationGame game = gc.createGame(left, right, pair.fst(), pair.snd(), maxLookahead, known);
                Set wrEven = this.solver.solve(game).playerEven();
                Set similar = wrEven.stream().filter(s -> s.owner().isOdd()).map(s -> Pair.of(s.odd(), s.even())).collect(Collectors.toSet());
                if (wrEven.contains(game.initialState())) {
                    assert (!similar.isEmpty());
                    known.addAll(similar);
                    seen.addAll(known);
                }
                return SimulationStats.of(System.currentTimeMillis() - startTime, game);
            }
            return null;
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        BuchiSimulation.logStats(stats);
        logger.fine("Obtained " + known.size() + " simulation pairs");
        return known;
    }

    private <S> Set<Pair<S, S>> multipebbleSimulate(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int pebbleCount, MultipebbleGameConstructor<S> gc) {
        assert (pebbleCount > 0);
        if (BuchiSimulation.automatonTrivial(left) || BuchiSimulation.automatonTrivial(right)) {
            return Set.of();
        }
        Set smallerRel = pebbleCount > 1 ? this.multipebbleSimulate(left, right, 1, gc) : Set.of();
        ConcurrentHashMap.KeySetView known = ConcurrentHashMap.newKeySet();
        known.addAll(smallerRel);
        ConcurrentHashMap.KeySetView seen = ConcurrentHashMap.newKeySet();
        List<SimulationStats> stats = Pair.allPairs(left.states(), right.states()).stream().filter(p -> !known.contains(p)).map(pair -> {
            if (seen.add(pair)) {
                long startTime = System.currentTimeMillis();
                SimulationGame game = gc.createGame(left, right, pair.fst(), pair.snd(), pebbleCount, known);
                Set wrEven = this.solver.solve(game).playerEven();
                Set similar = wrEven.stream().filter(s -> s.even().count() == 1 && s.owner().isOdd()).map(s -> Pair.of(s.odd().state(), s.even().onlyState())).collect(Collectors.toSet());
                if (wrEven.contains(game.initialState())) {
                    assert (!similar.isEmpty());
                    known.addAll(similar);
                    seen.addAll(similar);
                }
                return SimulationStats.of(System.currentTimeMillis() - startTime, game);
            }
            return null;
        }).filter(Objects::nonNull).collect(Collectors.toList());
        BuchiSimulation.logStats(stats);
        logger.fine("Obtained " + known.size() + " simulation pairs");
        if (pebbleCount > 1 && known.size() < smallerRel.size()) {
            throw new AssertionError((Object)"something went wrong!");
        }
        return known;
    }

    private static void logStats(Collection<SimulationStats> stats) {
        if (!stats.isEmpty()) {
            int avgSize = stats.stream().map(stat -> stat.graphSize).reduce(Integer::sum).orElse(0) / stats.size();
            long avgTime = stats.stream().map(stat -> stat.computationTime).reduce(Long::sum).get() / (long)stats.size();
            logger.fine("Solved " + stats.size() + " simulation games.");
            logger.fine("Average game size: " + avgSize);
            logger.fine("Average computation time: " + avgTime + "ms");
        }
    }

    private static <S> boolean automatonTrivial(Automaton<S, ? extends BuchiAcceptance> aut) {
        return aut.initialStates().isEmpty() || aut.initialStates().stream().allMatch(state -> aut.edges(state).isEmpty());
    }

    public <S> Set<Pair<S, S>> backwardSimulation(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int pebbleCount) {
        return this.multipebbleSimulate(left, right, pebbleCount, (l1, r1, red, blue, pc, known) -> new SimulationGame(new BackwardDirectSimulation<Object>(l1, r1, red, blue, pc, known)));
    }

    public <S> Set<Pair<S, S>> fairSimulation(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int pebbleCount) {
        return this.multipebbleSimulate(left, right, pebbleCount, (l1, r1, red, blue, pc, known) -> new SimulationGame(new ForwardFairSimulation<Object>(l1, r1, red, blue, pc, known)));
    }

    public <S> Set<Pair<S, S>> delayedSimulation(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int pebbleCount) {
        return this.multipebbleSimulate(left, right, pebbleCount, (left1, right1, red, blue, pebbleCount1, known) -> new SimulationGame(new ForwardDelayedSimulation<Object>(left1, right1, red, blue, pebbleCount1, known)));
    }

    public <S> Set<Pair<S, S>> directSimulation(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int pebbleCount) {
        return this.multipebbleSimulate(left, right, pebbleCount, (left1, right1, red, blue, pebbleCount1, known) -> new SimulationGame(new ForwardDirectSimulation<Object>(left1, right1, red, blue, pebbleCount1, known)));
    }

    public <S> Set<Pair<S, S>> directLookaheadSimulation(Automaton<S, ? extends BuchiAcceptance> left, Automaton<S, ? extends BuchiAcceptance> right, int maxLookahead) {
        return this.lookaheadSimulate(left, right, maxLookahead, (l1, r1, red, blue, ml, known) -> new SimulationGame(new ForwardDirectLookaheadSimulation<Object>(l1, r1, red, blue, ml, known)));
    }

    public <S> boolean directSimulates(Automaton<S, BuchiAcceptance> left, Automaton<S, BuchiAcceptance> right, S leftState, S rightState, int pebbleCount) {
        Set<Pair<S, S>> rel = this.directSimulation(left, right, pebbleCount);
        return rel.contains(Pair.of(leftState, rightState));
    }

    private static class SimulationStats {
        public final long computationTime;
        public final int graphSize;

        private SimulationStats(long ct, int gs) {
            this.computationTime = ct;
            this.graphSize = gs;
        }

        public static SimulationStats of(long time, SimulationGame<?, ?> game) {
            return new SimulationStats(time, game.states().size());
        }
    }

    private static interface LookaheadGameConstructor<S> {
        public SimulationGame<S, SimulationStates.LookaheadSimulationState<S>> createGame(Automaton<S, ? extends BuchiAcceptance> var1, Automaton<S, ? extends BuchiAcceptance> var2, S var3, S var4, int var5, Set<Pair<S, S>> var6);
    }

    private static interface MultipebbleGameConstructor<S> {
        public SimulationGame<S, SimulationStates.MultipebbleSimulationState<S>> createGame(Automaton<S, ? extends BuchiAcceptance> var1, Automaton<S, ? extends BuchiAcceptance> var2, S var3, S var4, int var5, Set<Pair<S, S>> var6);
    }

    public static enum SimulationType {
        DIRECT_SIMULATION_COLOUR_REFINEMENT,
        DIRECT_SIMULATION,
        DELAYED_SIMULATION,
        FAIR_SIMULATION,
        BACKWARD_SIMULATION,
        LOOKAHEAD_DIRECT_SIMULATION;

    }
}

